from typing import Generator
import httpx
import pydantic
import webbrowser

from server import start_server

class OIDCAuth(httpx.Auth, pydantic.BaseModel):
    """
    Implements OpenID Connect (OIDC) authorization flow for HTTP requests using `httpx.Auth`.

    Attributes:
        client_id (str): The client ID for the OIDC client.
        client_secret (str): The client secret for the OIDC client.
        discovery_url (str): The OIDC discovery URL to fetch the authorization and token endpoints.
        redirect_port (int): The port number for the local redirect URI (default is 8091).
        bearer_token (str | None): The bearer token obtained after the OIDC flow, used for authorization.
    """

    client_id: str
    client_secret: str
    discovery_url: str
    redirect_port: int = 8091
    bearer_token: str | None = None

    def auth_flow(self, request) -> Generator[httpx.Request, httpx.Response, None]:
        """
        Auth flow generator that attaches the authorization header to the HTTP request.
        If the bearer token is not set, it performs the OIDC flow to obtain one.

        Args:
            request (httpx.Request): The HTTP request object to which the authorization header will be attached.

        Yields:
            httpx.Request: The HTTP request with the authorization header set.
        """
        if self.bearer_token is not None:
            # Bearer token is already set
            request.headers['Authorization'] = f"Bearer {self.bearer_token}"
            yield request
            return

        # Run the OIDC flow
        authorization_endpoint, token_endpoint = self._get_endpoints()
        self._get_access_token(
            authorization_endpoint=authorization_endpoint,
            token_endpoint=token_endpoint,
        )

        request.headers['Authorization'] = f"Bearer {self.bearer_token}"

        yield request


    def _retrieve_button_href(self):
        pass

    def _get_endpoints(self):
        """
        Retrieves the authorization and token endpoints from the OIDC discovery URL.

        Returns:
            tuple: A tuple containing the authorization endpoint URL and the token endpoint URL.

        Raises:
            AssertionError: If the discovery response does not contain the required endpoints.
        """
        response = httpx.get(self.discovery_url)
        response.raise_for_status()
        provider_cfg = response.json()

        assert 'authorization_endpoint' in provider_cfg, "Authorization endpoint is missing"
        assert 'token_endpoint' in provider_cfg, "Token endpoint is missing"

        return (
            provider_cfg['authorization_endpoint'],
            provider_cfg['token_endpoint']
        )

    def _get_access_token(
        self,
        authorization_endpoint: str,
        token_endpoint: str,
    ) -> None:
        """
        Obtains the access token by exchanging the authorization code at the token endpoint.

        Args:
            authorization_endpoint (str): The authorization endpoint URL.
            token_endpoint (str): The token endpoint URL.

        Raises:
            Exception: If the access token could not be obtained.
        """
        authorization_token = self._get_authorization_token(
            authorization_endpoint=authorization_endpoint
        )

        token_request_body = {
            'grant_type': 'authorization_code',
            'code': authorization_token,
            'redirect_uri': f"http://localhost:{self.redirect_port}/callback",
            'client_id': self.client_id,
            'client_secret': self.client_secret,
        }

        assert token_endpoint is not None, "Token endpoint is not set"

        response = httpx.post(token_endpoint, data=token_request_body)
        response.raise_for_status()

        token_response = response.json()

        if 'access_token' not in token_response:
            raise Exception(f"Failed to get access token: {token_response}")

        self.bearer_token = token_response['access_token']

    def _get_authorization_token(self, authorization_endpoint: str) -> str:
        """
        Initiates the authorization code flow by opening the authorization URL in a web browser.

        Args:
            authorization_endpoint (str): The authorization endpoint URL.

        Returns:
            str: The authorization code received via the redirect URI.
        """
        auth_url = self._create_auth_url(authorization_endpoint)
        webbrowser.open(auth_url)
        return start_server(self.redirect_port)

    def _create_auth_url(self, authorization_endpoint: str) -> str:
        """
        Constructs the authorization URL with the required query parameters.

        Args:
            authorization_endpoint (str): The authorization endpoint URL.

        Returns:
            str: The complete authorization URL.
        """
        return f"{authorization_endpoint}?response_type=code&client_id={self.client_id}&redirect_uri=http://localhost:{self.redirect_port}/callback&scope=openid"
